<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Contact;
use App\Http\Resources\ContactResource;
use App\Http\Requests\Contact\StoreContactRequest;
use App\Http\Requests\Contact\UpdateContactRequest;
use App\Services\ContactService;
use App\Traits\ApiResponse;

class ContactController extends Controller
{
    use ApiResponse;

    protected ContactService $contactService;

    public function __construct(ContactService $contactService)
    {
        $this->contactService = $contactService;
    }

    public function index()
    {
        try {
            $contacts = $this->contactService->getPaginated();
            return $this->successResponse(
                ContactResource::collection($contacts)
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Error fetching contacts: ' . $e->getMessage());
        }
    }

    public function store(StoreContactRequest $request)
    {
        try {
            $contact = $this->contactService->create($request->validated());
            
            return $this->createdResponse(
                new ContactResource($contact),
                'Message sent successfully'
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Error sending message: ' . $e->getMessage(), 500);
        }
    }

    public function show(Contact $contact)
    {
        try {
            return $this->successResponse(
                new ContactResource($contact)
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Error fetching contact: ' . $e->getMessage());
        }
    }

    public function update(UpdateContactRequest $request, Contact $contact)
    {
        try {
            $contact = $this->contactService->markAsRead($contact);
            return $this->successResponse(
                new ContactResource($contact)
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Error updating contact: ' . $e->getMessage(), 500);
        }
    }

    public function destroy(Contact $contact)
    {
        try {
            $this->contactService->delete($contact);
            return $this->deletedResponse();
        } catch (\Exception $e) {
            return $this->errorResponse('Error deleting contact: ' . $e->getMessage(), 500);
        }
    }

    public function unread()
    {
        try {
            $contacts = $this->contactService->getUnread();
            return $this->successResponse(
                ContactResource::collection($contacts)
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Error fetching unread contacts: ' . $e->getMessage());
        }
    }
}
