<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Services\PortfolioSettingService;
use App\Traits\ApiResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class PortfolioSettingController extends Controller
{
    use ApiResponse;

    protected $settingService;

    public function __construct(PortfolioSettingService $settingService)
    {
        $this->settingService = $settingService;
        // $this->middleware('auth:sanctum');
    }

    public function index()
    {

        try {
            $settings = $this->settingService->getAllSettings();
            return $this->successResponse($settings);
        } catch (\Exception $e) {
            return $this->errorResponse('Error fetching settings: ' . $e->getMessage());
        }
    }

    public function show($key)
    {
        try {
            $setting = $this->settingService->getSetting($key);
            if (!$setting) {
                return $this->errorResponse('Setting not found', 404);
            }
            return $this->successResponse($setting);
        } catch (\Exception $e) {
            return $this->errorResponse('Error fetching setting: ' . $e->getMessage());
        }
    }

    public function update(Request $request, $key)
    {
        try {
            $validator = Validator::make($request->all(), [
                'value' => 'required',
                'type' => 'sometimes|string|in:string,integer,boolean,array,json',
                'description' => 'sometimes|string|max:255'
            ]);

            if ($validator->fails()) {
                return $this->errorResponse($validator->errors(), 422);
            }

            $setting = $this->settingService->updateSetting(
                $key,
                $request->value,
                $request->type ?? 'string',
                $request->description
            );

            return $this->successResponse($setting);
        } catch (\Exception $e) {
            return $this->errorResponse('Error updating setting: ' . $e->getMessage());
        }
    }

    public function bulkUpdate(Request $request)
    {
        try {
            $settings = $this->settingService->bulkUpdateSettings($request->all());
            return $this->successResponse($settings);
        } catch (\Exception $e) {
            return $this->errorResponse('Error updating settings: ' . $e->getMessage());
        }
    }

    public function uploadSettingImage(Request $request)
    {
        // Log::info($request->all());
        try {
            $validator = Validator::make($request->all(), [
                'key'   => 'required|string',
                'image' => 'required|file|mimes:jpeg,png,jpg,gif,webp,svg,ico|max:10240'
            ]);
            if ($validator->fails()) {
                return $this->errorResponse($validator->errors(), 422);
            }
            $imagePath = $this->settingService->uploadImage($request);
            return $this->successResponse(['image_url' => getFileUrl($imagePath)]);
        } catch (\Exception $e) {
            return $this->errorResponse('Error uploading image: '. $e->getMessage());
        }
    }

    public function destroy($key)
    {
        try {
            $this->settingService->deleteSetting($key);
            return $this->deletedResponse();
        } catch (\Exception $e) {
            return $this->errorResponse('Error deleting setting: ' . $e->getMessage());
        }
    }
}