<?php

namespace App\Services;

use App\Models\Article;
use Illuminate\Http\Request;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Facades\Log;

class ArticleService extends BaseService
{
    /**
     * Get paginated articles
     */
    public function getPaginatedArticles(Request $request): LengthAwarePaginator
    {
        $query = Article::with('category');

        if ($request->has('featured')) {
            $query->featured();
        }

        return $query->latest('published_at')
                    ->paginate(10);
    }

    /**
     * Create a new article
     */
    public function create(array $data): Article
    {
        if (isset($data['image'])) {
            $data['image'] = $this->uploadFile($data['image'], 'articles');
        }

        return Article::create($data);
    }

    /**
     * Update an existing article
     */
    public function update(Article $article, array $data): Article
    {
        Log::info('update article', $data);
        if (isset($data['image'])) {
            // $this->deleteFile($article->image);
            $data['image'] = $this->uploadFile($data['image'], 'articles', $article->image);
        } else {
            $data['image'] = $article->image;
        }

        $article->update($data);
        return $article;
    }

    /**
     * Delete an article
     */
    public function delete(Article $article): void
    {
        $this->deleteFile($article->image);
        $article->delete();
    }

    /**
     * Find article by slug
     */
    public function findBySlug(string $slug): Article
    {
        return Article::where('slug', $slug)
                    ->published()
                    ->firstOrFail();
    }
}