<?php

namespace App\Services;

use App\Models\PortfolioProject;
use App\Models\ProjectImage;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Http\UploadedFile;

class PortfolioService extends BaseService
{
    /**
     * Store a new portfolio project with its images
     */
    public function createProject(array $data, ?UploadedFile $thumbnail = null, array $images = [], array $captions = []): PortfolioProject
    {
        try {
            DB::beginTransaction();

            // Handle thumbnail upload
            if ($thumbnail) {
                $data['thumbnail'] = $this->uploadFile($thumbnail, 'portfolios/thumbnails');
            }

            // Create project
            $project = PortfolioProject::create($data);

            // Create features
            $features = $data['features'] ? json_decode($data['features']) : null;
   
            if ($features) {
                foreach ($features as $feature) {
                    $project->features()->create([
                        'feature' => $feature->feature,
                        'description' => $feature->description,
                    ]);
                }
            }

            // Create why choose us
            $whyChooseUs = $data['why_choose_us'] ? json_decode($data['why_choose_us']) : null;

            if (!empty($whyChooseUs)) {
                foreach ($whyChooseUs as $whyChoose) {
                    $project->whyChooseUs()->create([
                        'title' => $whyChoose->title,
                        'description' => isset($whyChoose->description) ? $whyChoose->description : null,
                        'display_order' => isset($whyChoose->display_order) ? $whyChoose->display_order : 0,
                    ]);
                }
            }

            if (!empty($data['testimonial_ids'])) {
                $project->testimonials()->sync($data['testimonial_ids']);
            } else {
                $project->testimonials()->detach(); // clear all if none selected
            }

            // Handle multiple images
            if (!empty($images)) {
                $this->attachImages($project, $images, $captions);
            }

            DB::commit();
            return $project;

        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Update an existing portfolio project
     */
    public function updateProject(PortfolioProject $project, array $data, ?UploadedFile $thumbnail = null, array $newImages = [], array $newCaptions = [], array $removeImageIds = []): PortfolioProject
    {
        try {
            DB::beginTransaction();

            // Handle thumbnail update
            if ($thumbnail) {
                $this->deleteFile($project->thumbnail);
                $data['thumbnail'] = $this->uploadFile($thumbnail, 'portfolios/thumbnails');
            }

            // Update project
            $project->update($data);

            // Update Features
            if (isset($data['features'])) {
                // Decode JSON if passed as string
                $features = is_string($data['features']) ? json_decode($data['features']) : $data['features'];

                // Delete old features
                $project->features()->delete();

                // Insert new features
                foreach ($features as $feature) {
                    $project->features()->create([
                        'feature' => $feature->feature, 
                        'description' => $feature->description
                    ]);
                }
            }

            //Update why choose us
            if (isset($data['why_choose_us'])) {
                $whyChooseUs = $data['why_choose_us'] ? json_decode($data['why_choose_us']) : null;
                if (!empty($whyChooseUs)) {
                    $project->whyChooseUs()->delete();
                    foreach ($whyChooseUs as $whyChoose) {
                        $project->whyChooseUs()->create([
                            'title' => $whyChoose->title,
                            'description' => isset($whyChoose->description) ? $whyChoose->description : null,
                            'display_order' => isset($whyChoose->display_order) ? $whyChoose->display_order : 0,
                        ]);
                    }
                }
            }

            if (!empty($data['testimonial_ids'])) {
                $project->testimonials()->sync($data['testimonial_ids']);
            } else {
                $project->testimonials()->detach(); // clear all if none selected
            }

            // Remove specified images
            if (!empty($removeImageIds)) {
                $this->removeImages($project, $removeImageIds);
            }

            // Add new images
            if (!empty($newImages)) {
                $this->attachImages($project, $newImages, $newCaptions);
            }

            DB::commit();
            return $project;

        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Delete a portfolio project and all associated images
     */
    public function deleteProject(PortfolioProject $project): void
    {
        try {
            DB::beginTransaction();

            // Delete thumbnail
            $this->deleteFile($project->thumbnail);

            // Delete all associated images
            foreach ($project->images as $image) {
                $this->deleteFile($image->image_url);
            }
            
            $project->images()->delete();
            $project->features()->delete();
            $project->whyChooseUs()->delete();
            $project->delete();

            DB::commit();

        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Attach multiple images to a project
     */
    protected function attachImages(PortfolioProject $project, array $images, array $captions): void
    {
        foreach ($images as $index => $image) {
            $path = $this->uploadFile($image, 'portfolios/images');
            $project->images()->create([
                'image_url' => $path,
                'caption' => $captions[$index] ?? null
            ]);
        }
    }

    /**
     * Remove specified images from a project
     */
    protected function removeImages(PortfolioProject $project, array $imageIds): void
    {
        $imagesToRemove = ProjectImage::whereIn('id', $imageIds)
            ->where('project_id', $project->id)
            ->get();

        foreach ($imagesToRemove as $image) {
            $this->deleteFile($image->image_url);
            $image->delete();
        }
    }
} 