<?php

namespace App\Services;

use App\Models\PortfolioSetting;
use Illuminate\Support\Facades\Cache;
use Illuminate\Http\Request;

class PortfolioSettingService extends BaseService
{
    protected $model;
    protected const CACHE_PREFIX = 'portfolio_settings:';
    protected const CACHE_TTL = 86400; // 24 hours

    public function __construct(PortfolioSetting $model)
    {
        $this->model = $model;
    }

    protected function getCacheKey($key = null)
    {
        return static::CACHE_PREFIX . ($key ? $key : 'all');
    }

    public function getAllSettings()
    {
        return Cache::remember($this->getCacheKey(), static::CACHE_TTL, function () {
            return $this->model->all()->pluck('value', 'key');
        });
    }

    public function getSetting($key, $default = null)
    {
        return Cache::remember($this->getCacheKey($key), static::CACHE_TTL, function () use ($key, $default) {
            return $this->model->getSetting($key, $default);
        });
    }

    public function updateSetting($key, $value, $type = 'string', $description = null)
    {
        //check if key exists
        $setting = $this->model->where('key', $key)->first();
        if ($setting) {
            $setting->value = $value;
            $setting->type = $type;
            $setting->description = $description;
            $setting->save();
        } else {
            $setting = $this->model->create([
                'key' => $key,
                'value' => $value,
                'type' => $type,
                'description' => $description
            ]);
        }

        
        // Invalidate both specific and all settings cache
        Cache::forget($this->getCacheKey($key));
        Cache::forget($this->getCacheKey());
        
        return $setting;
    }

    public function deleteSetting($key)
    {
        $result = $this->model->where('key', $key)->delete();
        
        // Invalidate both specific and all settings cache
        Cache::forget($this->getCacheKey($key));
        Cache::forget($this->getCacheKey());
        
        return $result;
    }

    public function bulkUpdateSettings(array $settings)
    {
        $updated = [];
        foreach ($settings as $key => $value) {
            $updated[$key] = $this->updateSetting($key, $value);
        }
        
        // Invalidate all settings cache
        Cache::forget($this->getCacheKey());
        
        return $updated;
    }

    public function uploadImage(Request $request): string
    {
        $file = $request->file('image');
        $key = $request->key;

        $setting = PortfolioSetting::where('key',$key)->pluck('value', 'key')->toArray();

        $imagePath = null;
        $previousPath = array_key_exists($key, $setting) ? $setting[$key] : null;
        $imagePath = $this->uploadFile($file, 'settings', $previousPath);

        $this->updateSetting(
            $key, $imagePath,
        );

        // Invalidate both specific and all settings cache
        Cache::forget($this->getCacheKey($key));
        Cache::forget($this->getCacheKey());

        return $imagePath;
    }
}